unit DataBindingWriteU;

{
  Demonstrate the generation of an XML document from a database
  using the XML Data Binding abilities of Delphi 6 and 7.
  Requires 'movie-watcher' alias to be set up in BDE.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 27 November, 2001.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Db, DBTables, StdCtrls, ExtCtrls, XMLIntf, DataBindingObjs;

type
  TfrmXDBXML = class(TForm)
    memXML: TMemo;
    pnlButtons: TPanel;
      btnGenerate: TButton;
      btnSave: TButton;
    dlgSave: TSaveDialog;
    qryMovie: TQuery;
      qryMovieMovie_id: TIntegerField;
      qryMovieName: TStringField;
      qryMovieRating: TStringField;
      qryMovieLength_Mins: TIntegerField;
      qryMovieDirector: TStringField;
      qryMovieSynopsis: TMemoField;
      qryMovieURL: TStringField;
      qryMovieLogo_URL: TStringField;
    dsrMovie: TDataSource;
    qryStars: TQuery;
      qryStarsStar_id: TIntegerField;
      qryStarsMovie_id: TIntegerField;
      qryStarsStar: TStringField;
    qryCinema: TQuery;
      qryCinemaCinema_id: TIntegerField;
      qryCinemaName: TStringField;
      qryCinemaPhone: TStringField;
      qryCinemaAddress: TStringField;
      qryCinemaDirections: TMemoField;
      qryCinemaCandy_bar: TBooleanField;
      qryCinemaDisabled_access: TBooleanField;
    dsrCinema: TDataSource;
    qryPricing: TQuery;
      qryPricingPricing_id: TIntegerField;
      qryPricingCinema_id: TIntegerField;
      qryPricingName: TStringField;
      qryPricingPeriod: TStringField;
      qryPricingAdult: TFloatField;
      qryPricingChild: TFloatField;
      qryPricingDiscount: TFloatField;
    qryScreening: TQuery;
      qryScreeningMovie_id: TIntegerField;
      qryScreeningCinema_id: TIntegerField;
      qryScreeningStart_date: TDateField;
      qryScreeningEnd_date: TDateField;
      qryScreeningDigital_sound: TStringField;
      qryScreeningNo_passes: TBooleanField;
    dsrScreening: TDataSource;
    qrySessions: TQuery;
      qrySessionsMovie_id: TIntegerField;
      qrySessionsCinema_id: TIntegerField;
      qrySessionsTime: TTimeField;
      qrySessionsPricing_id: TIntegerField;
    procedure btnGenerateClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure GetMemoText(Sender: TField; var Text: String;
      DisplayText: Boolean);
  private
  public
  end;

var
  frmXDBXML: TfrmXDBXML;

implementation

{$R *.DFM}

uses
  CommonXML;

{ Supply memo as text }
procedure TfrmXDBXML.GetMemoText(Sender: TField; var Text: String;
  DisplayText: Boolean);
begin
  Text := Sender.AsString;
end;

{ Generate the XML document as text }
procedure TfrmXDBXML.btnGenerateClick(Sender: TObject);
var
  MWObjs: IXMLMovieWatcherType;

  { Generate comment and style sheet reference }
  procedure GenerateHeaders;
  begin
    with MWObjs.OwnerDocument do
    begin
      Options := Options + [doNodeAutoIndent]; 
      ChildNodes.Insert(1, CreateNode(XMLComment, ntComment));
      ChildNodes.Insert(2,
        CreateNode(XMLStyleTag, ntProcessingInstr, XMLStyleAttrs));
    end;
  end;

  { Generate elements for each movie }
  procedure GenerateMovies;
  var
    Movie: IXMLMovieType;
  begin
    with qryMovie do
    begin
      First;
      while not EOF do
      begin
        Movie           := MWObjs.Movies.Add;
        Movie.Id        := FieldByName(MovieIdField).DisplayText;
        Movie.Rating    := FieldByName(RatingField).DisplayText;
        if FieldByName(LogoURLField).DisplayText <> '' then
          Movie.LogoUrl := FieldByName(LogoURLField).DisplayText;
        if FieldByName(URLField).DisplayText <> '' then
          Movie.Url     := FieldByName(URLField).DisplayText;
        Movie.Name      := FieldByName(NameField).DisplayText;
        Movie.Length    := FieldByName(LengthField).AsInteger;
        Movie.Director  := FieldByName(DirectorField).DisplayText;
        { Add the stars }
        with qryStars do
        begin
          First;
          while not EOF do
          begin
            Movie.Starring.Add(FieldByName(StarField).DisplayText);
            Next;
          end;
        end;
        Movie.Synopsis  := FieldByName(SynopsisField).DisplayText;
        Next;
      end;
    end;
  end;

  { Generate elements for each cinema }
  procedure GenerateCinemas;
  var
    Cinema: IXMLCinemaType;
    Price: IXMLPricesType;
  begin
    with qryCinema do
    begin
      First;
      while not EOF do
      begin
        Cinema                     := MWObjs.Cinemas.Add;
        Cinema.Id                  := FieldByName(CinemaIdField).DisplayText;
        Cinema.Name                := FieldByName(NameField).DisplayText;
        Cinema.Phone               := FieldByName(PhoneField).DisplayText;
        Cinema.Address             := FieldByName(AddressField).DisplayText;
        Cinema.Directions          := FieldByName(DirectionsField).DisplayText;
        Cinema.Facilities.CandyBar := FieldByName(CandyBarField).AsBoolean;
        Cinema.Facilities.DisabledAccess :=
          FieldByName(DisabledField).AsBoolean;
        { Add the pricing schemes }
        with qryPricing do
        begin
          First;
          while not EOF do
          begin
            Price          := Cinema.Pricing.Add;
            Price.Id       := FieldByName(PricingIdField).DisplayText;
            Price.Name     := FieldByName(NameField).DisplayText;
            Price.Period   := FieldByName(PeriodField).DisplayText;
            Price.Adult    := FieldByName(AdultField).AsFloat;
            Price.Child    := FieldByName(ChildField).AsFloat;
            Price.Discount := FieldByName(DiscountField).AsFloat;
            Next;
          end;
        end;
        Next;
      end;
    end;
  end;

  { Generate elements for each screening }
  procedure GenerateScreenings;
  var
    Screening: IXMLScreeningType;
    Session: IXMLSessionType;
  begin
    with qryScreening do
    begin
      First;
      while not EOF do
      begin
        Screening           := MWObjs.Screenings.Add;
        Screening.MovieId   := FieldByName(MovieIdField).DisplayText;
        Screening.CinemaId  := FieldByName(CinemaIdField).DisplayText;
        Screening.StartDate := FieldByName(StartDateField).AsDateTime;
        Screening.EndDate   := FieldByName(EndDateField).AsDateTime;
        if FieldByName(DigSoundField).DisplayText <> '' then
          Screening.Features.DigitalSound :=
            FieldByName(DigSoundField).DisplayText;
        if FieldByName(NoPassesField).AsBoolean then
          Screening.Restrictions.NoPasses := True;
        { Add the session times }
        with qrySessions do
        begin
          First;
          while not EOF do
          begin
            Session         := Screening.Sessions.Add;
            Session.PriceId := FieldByName(PricingIdField).DisplayText;
            Session.Time    := FieldByName(TimeField).AsDateTime;
            Next;
          end;
        end;
        Next;
      end;
    end;
  end;

begin
  try
    Screen.Cursor := crHourglass;
    { Instantiate the XML binding }
    MWObjs        := NewMovieWatcher;
    { Generate the structure }
    GenerateHeaders;
    GenerateMovies;
    GenerateCinemas;
    GenerateScreenings;
    { And convert to XML }
    memXML.Lines  := MWObjs.OwnerDocument.XML;
  finally
    { Release the XML binding }
    MWObjs        := nil;
    Screen.Cursor := crDefault;
  end;
end;

{ Save the generated XML }
procedure TfrmXDBXML.btnSaveClick(Sender: TObject);
begin
  with dlgSave do
    if Execute then
      memXML.Lines.SaveToFile(Filename);
end;

end.
